//  Functions for the MFAMD model for binary, ordinal, nominal and continuous data. //
//////////////////////////////////////////////////////////////////////////////////////

#include<stdio.h>
#include<strings.h>
#include<stdlib.h>
#include<math.h>
#include<time.h>
#include<gsl/gsl_randist.h>
#include<gsl/gsl_cdf.h>

#include "Mem.h"
#include "FileOps.h"
#include "LinAlg.h"
#include "RandVarSim.h"
#include "MiscCFuncs.h"
#include "MFAMD_structs.h"
#include "MFAMD_Funcs.h"
#include "StatsBase.h"

// Function to sample the latent trait vector for all observations.
void ThetaFunc(int Q, int D, int N, int G, double **Iq, struct params *pars){
	int i, j, g, d;
	double **Mu_ig = TwodDoubleMemAlloc(N, D);
	double **thet_var = TwodDoubleMemAlloc(Q, Q);
	int *indx = OnedIntMemAlloc(Q);
	
	for (i=0; i<N; i++){
		double *mu_g = RowCol( pars->lambda[(int)pars->ind[i]], D, (Q+1), 0, 2 );
		for(d=0; d<D; d++)
			Mu_ig[i][d] = mu_g[d];
		free(mu_g);
	} // i
	
	for(g=0; g<G; g++){
		// Covariance calculation
		double **AT = MatTrans(pars->A[g], D, Q);
		double **ATiPs = DoubleMatMul(AT, Q, D, pars->inv_psi, D, D);
		double **ATiPsA = DoubleMatMul(ATiPs, Q, D, pars->A[g], D, Q);
		double **ATiPsAIq = DoubleMatAdd(ATiPsA, Q, Q, Iq, Q, Q);
		migs(ATiPsAIq, Q, thet_var, indx);

		// Mean calculation
		double **ZlessMu = DoubleMatSub(pars->Z, N, D, Mu_ig, N, D);
		double **ZlessMuT = MatTrans(ZlessMu, N, D);
		double **ATiPsMu = DoubleMatMul(ATiPs, Q, D, ZlessMuT, D, N);
		double **thet_mean_temp = DoubleMatMul(thet_var, Q, Q, ATiPsMu, Q, N);

		double **RMVN = TwodDoubleMemAlloc(pars->ng[g], Q);
		rmvnorm(pars->ng[g], Q, thet_var, RMVN);

		double **thet_mean = MatTrans(thet_mean_temp, Q, N);

		int c = 0;
		for(i=0; i<N; i++){
			if(pars->ind[i]==g){
				double *theta_i_temp = DoubleVecAdd(RMVN[c], Q, thet_mean[i], Q);
				for(j=0; j<Q; j++)
					pars->theta[i][j] = theta_i_temp[j];
				c++;
				free(theta_i_temp);
			}
		} //i
/*		freeDoubleMat(A_g, D);*/
		freeDoubleMat(AT, Q);
		freeDoubleMat(ATiPs, Q);
		freeDoubleMat(ATiPsA, Q);
		freeDoubleMat(ATiPsAIq, Q);
		freeDoubleMat(ZlessMu, N);
		freeDoubleMat(ZlessMuT, D);
		freeDoubleMat(ATiPsMu, Q);
		freeDoubleMat(thet_mean_temp, Q);
		freeDoubleMat(RMVN, pars->ng[g]);
		freeDoubleMat(thet_mean, N);
	}//g
	freeDoubleMat(Mu_ig, N);
	freeDoubleMat(thet_var, Q);
	free(indx);
}

// Funciton to sample the mixing weights.
void PiFunc(int G, struct priors *prior, struct params *pars){
	double *delta = DoubleVecAdd(pars->ng, G, prior->alpha, G);	
	rdirichlet(G, delta, pars->pi_vec);
	
	free(delta);
}

// Function to allocate observations to groups.
/*void AllocFunc(int D, int N, int G, double ***Mu, double **Z, double **inv_psi, double *pi_vec, int **Alloc){*/
/*	int g, i, d;*/
/*	double **Hold = TwodDoubleMemAlloc(N, G);*/
/*	double **Prob = TwodDoubleMemAlloc(N, G);*/
/*	double *sum_i = OnedDoubleMemAlloc(N);*/
/*	double jLogProd;*/
/*	for(i=0; i<N; i++)*/
/*		sum_i[i] = 0;*/

/*	for(g=0; g<G; g++){*/
/*		for (i=0; i<N; i++){*/
/*			jLogProd = 0;*/
/*			for(d=0; d<D; d++)*/
/*				jLogProd += log( gsl_ran_gaussian_pdf( (Z[i][d] - Mu[g][i][d]), (1/inv_psi[d][d])) );*/
/*				*/
/*			Hold[i][g] = log(pi_vec[g]) + jLogProd;*/
/*			sum_i[i] += exp(Hold[i][g]);*/
/*		} //i*/
/*	} //g*/
/*	*/
/*	for(i=0; i<N; i++){*/
/*		for(g=0; g<G; g++)*/
/*			Prob[i][g] = exp(Hold[i][g])/sum_i[i];*/

/*		rmultinom(1, Prob[i], G, Alloc[i]);	*/
/*	} //i*/
/*	freeDoubleMat(Hold, N);*/
/*	freeDoubleMat(Prob, N);*/
/*	free(sum_i);*/
/*}*/

/*void AllocFunc(int D, int N, int G, double *LL, struct params *pars){*/
/*	int g, i, d, j;*/
/*	double **Hold = TwodDoubleMemAlloc(N, G);*/
/*	double **Prob = TwodDoubleMemAlloc(N, G);*/
/*	double *sum_i = OnedDoubleMemAlloc(N);*/
/*	double *LL_temp = OnedDoubleMemAlloc(N);*/
/*	double jLogProd;*/
/*	*/
/*	for(i=0; i<N; i++)*/
/*		sum_i[i] = 0;*/

/*	for(g=0; g<G; g++){*/
/*		for (i=0; i<N; i++){*/
/*			jLogProd = 0;*/
/*			for(d=0; d<D; d++)*/
/*				jLogProd += log( gsl_ran_gaussian_pdf( (pars->Z[i][d] - pars->Mu[g][i][d]), sqrt(1/pars->inv_psi[d][d])) );*/
/*				*/
/*			Hold[i][g] = log(pars->pi_vec[g]) + jLogProd;*/
/*			sum_i[i] += exp(Hold[i][g]);*/
/*		} //i*/
/*	} //g*/
/*	*/
/*	for(i=0; i<N; i++){*/
/*		for(g=0; g<G; g++)*/
/*			Prob[i][g] = exp(Hold[i][g])/sum_i[i];*/
/*			printf("%lf \t \n",jLogProd);*/

/*			for(j=0; j<G; j++)*/
/*				printf("%d \t %lf \n",i, Prob[i][j]);*/

/*		rmultinom(1, Prob[i], G, pars->Alloc[i]);	*/

/*		LL_temp[i] = log(sum_i[i]);*/
/*	} //i*/
/*	LL[0]	= sum(LL_temp, N);*/

/*	freeDoubleMat(Hold, N);*/
/*	freeDoubleMat(Prob, N);*/
/*	free(sum_i);*/
/*	free(LL_temp);*/
/*}*/

// More stable version of AllocFunc to deal with v.small numbers
void AllocFunc(int D, int N, int G, struct params *pars){
	int g, i, d;
	double *Hold = OnedDoubleMemAlloc(G);
	double *Prob = OnedDoubleMemAlloc(G);
	double jLogProd;
	double log_sum;
	
	for (i=0; i<N; i++){
		for(g=0; g<G; g++){
			jLogProd = 0;
			for(d=0; d<D; d++)
				jLogProd += log( gsl_ran_gaussian_pdf( (pars->Z[i][d] - pars->Mu[g][i][d]), sqrt(1/pars->inv_psi[d][d])) );
				
			Hold[g] = log(pars->pi_vec[g]) + jLogProd;
		} //g

		log_sum = logsum(Hold, G);

		for(g=0; g<G; g++)
			Prob[g] = exp(Hold[g] - log_sum);

		rmultinom(1, Prob, G, pars->Alloc[i]);	
	} //i

	free(Hold);
	free(Prob);
}

// Funtion to sample the loadings matrix. Requires a prior mean vector of length (Q+1) and a prior covaraince matrix of dimension (Q+1) x (Q+1).
void LambdaFunc(int D, int Q, int N, int G, struct priors *prior, struct params *pars){
	int i, g, d, q, c;
	double **lambda_var = TwodDoubleMemAlloc((Q+1), (Q+1));
	int *indx = OnedIntMemAlloc(Q+1);

	for(g=0; g<G; g++){
		double **theta_til_g = TwodDoubleMemAlloc(pars->ng[g], (Q+1));
		double *Z_gd = OnedDoubleMemAlloc(pars->ng[g]);
		
		c = 0;
		for(i=0; i<N; i++){
			if(pars->ind[i]==g){
				for(q=0; q<(Q+1); q++)
					theta_til_g[c][q] = pars->theta_til[i][q];
				c++;
			}
		} //i

		for(d=0; d<D; d++){
			c = 0;
			for(i=0; i<N; i++){
				if(pars->ind[i]==g){
					Z_gd[c] = pars->Z[i][d];
					c++;
				}
			}// i
			
			double **ThT = MatTrans(theta_til_g, pars->ng[g], (Q+1));
			double **iPsThT = DoubleScalarMatMul(ThT, (Q+1), pars->ng[g], pars->inv_psi[d][d] );
			double **iPsThTTh = DoubleMatMul( iPsThT, (Q+1), pars->ng[g], theta_til_g, pars->ng[g], (Q+1) );			
			double **iPsThTThplus = DoubleMatAdd(iPsThTTh, (Q+1), (Q+1), prior->invSig_lam, (Q+1), (Q+1));
			migs( iPsThTThplus, (Q+1), lambda_var, indx );

			double *tempVec1 = DoubleMatVecMul(iPsThT, (Q+1), pars->ng[g], Z_gd, pars->ng[g], 0);
			double *tempVec2 = DoubleMatVecMul(prior->invSig_lam, (Q+1), (Q+1), prior->mu_lam, (Q+1), 0);
			double *tempAdd = DoubleVecAdd( tempVec1, (Q+1), tempVec2, (Q+1) );
			double *lambda_mean = DoubleMatVecMul(lambda_var, (Q+1), (Q+1), tempAdd, (Q+1), 0);
			double *mvn_dev = OnedDoubleMemAlloc((Q+1));
			rmvnorm_1d((Q+1), lambda_var, mvn_dev);
			double *lambda_temp = DoubleVecAdd( mvn_dev, (Q+1), lambda_mean, (Q+1));
			for(q=0; q<(Q+1); q++)
				pars->lambda[g][d][q] = lambda_temp[q];
			
			freeDoubleMat(ThT, (Q+1));
			freeDoubleMat(iPsThT, (Q+1));
			freeDoubleMat(iPsThTTh, (Q+1));
			freeDoubleMat(iPsThTThplus, (Q+1));
			free(tempVec1);
			free(tempVec2);
			free(tempAdd);
			free(lambda_mean);
			free(mvn_dev);
			free(lambda_temp);
		} //d
		freeDoubleMat(theta_til_g, pars->ng[g]);
		free(Z_gd);
	} //g
	freeDoubleMat(lambda_var, (Q+1));
	free(indx);	
}

// Function to generate the latent data.
double *ZNomSim(double *vec, int y, double *vecMean, double vecVar, int vecLength){
	double *new_vec = OnedDoubleMemAlloc(vecLength);
	int i;

	if(y==1){
		for(i=0; i<vecLength; i++)
			new_vec[i] = rtruncn (vecMean[i], vecVar, -HUGE_VAL, 0.0);	
	}else{
		double max_vec = max(vec, vecLength);
		for(i=0; i<vecLength; i++){
			if( (i+2)!=y ){
				new_vec[i] = rtruncn( vecMean[i], vecVar, -HUGE_VAL, max_vec );
			}else{
				new_vec[i] = 0;
			}
		} // i
		double max_new_vec = max(new_vec, vecLength);
		new_vec[y-2] = rtruncn( vecMean[y-2], vecVar, max_new_vec, HUGE_VAL );
	}
	return(new_vec);
	free(new_vec);
}

void ZFunc(int N, int D, int CnsIndx, int OrdIndx, double **GamMat, int nNom, double *Knom, int *nom_D, int *nom_J, struct params *pars){
	int i, d, k, n;
	double **Zmean = TwodDoubleMemAlloc(N, D);
	for(i=0; i<N; i++){
/*		double **Mu_i = Sheet(Mu, N, D, ind[i]);*/
		for(d=0; d<D; d++)
			Zmean[i][d] = pars->Mu[(int)pars->ind[i]][i][d];
/*		freeDoubleMat(Mu_i, N);*/
	}
/*	printf("Zmean:%lf \t %lf \n", Zmean[0][3], Zmean[0][7]);*/
/*	printf("Mu:%lf \t %lf \n", Mu[0][3][0], Mu[0][7][0]);*/

	if (CnsIndx != OrdIndx){
		// Ordinal Items
		for(d=CnsIndx; d<OrdIndx; d++){
			for (i=0; i<N; i++)
				pars->Z[i][d] = rtruncn( Zmean[i][d], 1, GamMat[d][(int)(pars->Y[i][d])-1], GamMat[d][(int)(pars->Y[i][d])] );
		}// d
	} // if
	
	if(OrdIndx != D){
		// Nominal Items
		for(n=0; n<nNom; n++){
			double *vec = OnedDoubleMemAlloc((int)Knom[n]-1);
			double *vecMean = OnedDoubleMemAlloc((int)Knom[n]-1);
		
			for(i=0; i<N; i++){
				for(k=0; k<(Knom[n]-1); k++){	
					vec[k] = pars->Z[i][(nom_D[n] + k)];
					vecMean[k] = Zmean[i][(nom_D[n] + k)];
				}// k

				double *ZHold = ZNomSim( vec, (int)pars->Y[i][nom_J[n]], vecMean, 1, ((int)Knom[n]-1) );

				for(k=0; k<(Knom[n]-1); k++)
					pars->Z[i][nom_D[n] + k] = ZHold[k];
			
				free(ZHold);
			}// i
			free(vec);
			free(vecMean);
		}// n
	}// if
	freeDoubleMat(Zmean, N);
}

// Function to generate the threshold parameters for the ordinal items
void GammaFunc(int N, int D, int nOrd, int CnsIndx, int OrdIndx, double *Kord, double *sig_MH, int *MHcount, double **GamMat, struct params *pars){
	int i, j, k;
	double **Gam_Mean = TwodDoubleMemAlloc(N, nOrd);
	double R, temp;

	for(i=0; i<N; i++){
		for(j=CnsIndx; j<OrdIndx; j++)
			Gam_Mean[i][j-CnsIndx] = pars->Mu[(int)pars->ind[i]][i][j];
	}// i

	for(j=0; j<nOrd; j++){
		double *u = OnedDoubleMemAlloc(Kord[j]+1);
		double *R1 = OnedDoubleMemAlloc(N);
		double *R2 = OnedDoubleMemAlloc(Kord[j]-2);
		if((int)Kord[j]==2){
			R = 0.0;
		}else{
			u[0] = -HUGE_VAL;
			u[1] = 0;
			u[(int)Kord[j]] = HUGE_VAL;

			for(k=2; k<Kord[j]; k++){
				u[k] = rtruncn( GamMat[CnsIndx+j][k], sig_MH[j], u[k-1], GamMat[CnsIndx+j][k+1]); 
				if(u[k] < u[k-1]){
					printf("Gamma proposal not increasing!");
				}
			} //k

			for(i=0; i<N; i++){	
				R1[i] = log( gsl_cdf_gaussian_P( (u[ (int)pars->Y[i][CnsIndx+j] ] - Gam_Mean[i][j]), 1) - gsl_cdf_gaussian_P( (u[ (int)pars->Y[i][CnsIndx+j]-1 ] - Gam_Mean[i][j]), 1) ) - log( gsl_cdf_gaussian_P( (GamMat[CnsIndx+j][(int)pars->Y[i][CnsIndx+j]] - Gam_Mean[i][j]), 1) - gsl_cdf_gaussian_P( (GamMat[CnsIndx+j][(int)pars->Y[i][CnsIndx+j]-1] - Gam_Mean[i][j]), 1) );
			}

			for(k=2; k<Kord[j]; k++)	
				R2[k-2] = log( gsl_cdf_gaussian_P((GamMat[CnsIndx+j][k+1] - GamMat[CnsIndx+j][k])/sig_MH[j], 1) - gsl_cdf_gaussian_P((u[k-1] - GamMat[CnsIndx+j][k])/sig_MH[j], 1) ) - log( gsl_cdf_gaussian_P((u[k+1] - u[k])/sig_MH[j], 1) - gsl_cdf_gaussian_P((GamMat[CnsIndx+j][k-1] - u[k])/sig_MH[j], 1) );

		R = sum(R1, N) + sum(R2, Kord[j]-2);
		} // if

		temp = runif(0, 1);
		if(log(temp)<R){
			for(k=2; k<Kord[j]; k++)
				GamMat[CnsIndx+j][k] = u[k];

			MHcount[j]++;
		} // if
		free(R1);
		free(R2);
		free(u);
	} // j
	freeDoubleMat(Gam_Mean, N);
}

void PsiFunc(int CnsIndx, int N, int D, struct priors *prior, struct params *pars){
	int i, j;
	
	for(j=0; j<CnsIndx; j++){	
		double *Mu_gj = OnedDoubleMemAlloc(N);
		for (i=0; i<N; i++){
			Mu_gj[i] = pars->Mu[(int)pars->ind[i]][i][j];
		} // i

		double *Z_j = RowCol(pars->Z, N, D, j, 2);
		double *ZMu = DoubleVecSub(Z_j, N, Mu_gj, N);
		double ZMu2 = DotProd(ZMu, N, ZMu, N);
		
		pars->inv_psi[j][j] = rgamma( (prior->beta1 + (N/2)), (1/(prior->beta2 + 0.5*ZMu2)) );
		
		free(Mu_gj);
		free(Z_j);
		free(ZMu);
	} // j
}

/* Function to sample all parameters from their full conditional distributions */
void SampleParameters(int G, int D, int Q, int N, int CnsIndx, int OrdIndx, int nOrd, double *Kord, double *Knom, double *sig_MH, int *MHcount, double **GamMat, double **Iq, int nNom, int *nom_D, int *nom_J, struct params *pars, struct priors *prior){
		int g, j, d, i;

		// Update Mu array.
		for(g=0; g<G; g++){
			double **Tlam_g = MatTrans(pars->lambda[g], D, (Q+1));
			double **MuTemp = DoubleMatMul(pars->theta_til, N, (Q+1), Tlam_g, (Q+1), D );
			for(i=0; i<N; i++){
				for(d=0; d<D; d++)
					pars->Mu[g][i][d] = MuTemp[i][d];
			} //i
			freeDoubleMat(MuTemp, N);
			freeDoubleMat(Tlam_g, (Q+1));
		} //g


		// Allocate individuals to groups and tabulate.
/*		AllocFunc(D, N, G, LL, pars);*/
		AllocFunc(D, N, G, pars);
		map(pars->Alloc, N, G, pars->ind);
		table(pars->ind, G, N, pars->ng);


		/////////////////////////////
/*		for(g=0; g<G; g++){*/
/*			printf("%lf ", pars->ng[g]);*/
/*		}*/
/*		printf("\n");*/
		///////////////////////////////

		// Sample mixing weights.
		PiFunc(G, prior, pars);

		// Sample threshold parameters if required.
		if(CnsIndx != OrdIndx)	
			GammaFunc(N, D, nOrd, CnsIndx, OrdIndx, Kord, sig_MH, MHcount, GamMat, pars);
		
		// Sample latent data.
		if(CnsIndx != D)	
			ZFunc(N, D, CnsIndx, OrdIndx, GamMat, nNom, Knom, nom_D, nom_J, pars);

		// Sample psi uniqueness matrix.
		if(CnsIndx != 0)
			PsiFunc(CnsIndx, N, D, prior, pars);

		// Sample latent traits.
		ThetaFunc(Q, D, N, G, Iq, pars);

		for(i=0; i<N; i++){
			pars->theta_til[i][0] = 1;
			for(j=0; j<Q; j++)
				pars->theta_til[i][j+1] = pars->theta[i][j];	
		} // i

		// Sample loadings matrices.
		LambdaFunc(D, Q, N, G, prior, pars);
		for (g=0; g<G; g++){
			for(d=0; d<D; d++){
				for (j=1; j<(Q+1); j++)
					pars->A[g][d][j-1] = pars->lambda[g][d][j];
			} // d
		} // g
}

/* Function to determine which variables should be dropped */
void VarSel(int G, int D, int N, int CnsIndx, int OrdIndx, double PropTol_cns, double PropTol_cat, int J, double *Knom, int *nom_D, struct params *pars, struct variableSelect *varSelect){
	int i, j;
	// Calculate empirical group means
	double **Zbar = TwodDoubleMemAlloc(G, D);
	for(i=0; i<N; i++){
		double *Ztemp = DoubleScalarVecMul(pars->Z[i], D, 1/(pars->ng[(int)pars->ind[i]]));
		double *Zbar_temp = DoubleVecAdd(Zbar[(int)pars->ind[i]], D, Ztemp, D);
		DoubleVecCopy(Zbar_temp, D, Zbar[(int)pars->ind[i]]);
		free(Ztemp);
		free(Zbar_temp);
	} // i

	// Within group sum of squared differences
	double **diff2_w = TwodDoubleMemAlloc(N, D);
	for(i=0; i<N; i++){
		double *diff_w_i = DoubleVecSub(pars->Z[i], D, Zbar[(int)pars->ind[i]], D);
		double *diff2_w_i = DoubleVecMul(diff_w_i, D, diff_w_i, D);
		DoubleVecCopy(diff2_w_i, D, diff2_w[i]);
		free(diff_w_i);
		free(diff2_w_i);
	} // i

	double (*sum_ptr)(double *, int);
	sum_ptr = &sum;
	double *Sdiff2_w = apply(diff2_w, N, D, 2, sum_ptr);
/*	DoubleVec2Screen(Sdiff2_w, D);*/

	// Overall sum of squared differences
	double (*var_ptr)(double *, int);
	var_ptr = &varVec;

	double *var_o = apply(pars->Z, N, D, 2, var_ptr);
	double *Sdiff2_o = DoubleScalarVecMul(var_o, D, (double)(N-1));
/*	DoubleVec2Screen(Sdiff2_o, D);*/
	
	// Proportion of overall variance which is within group variance (All D dims).
	double *vpd = DoubleVecDiv(Sdiff2_w, D, Sdiff2_o, D);
	DoubleVecCopy(vpd, D, varSelect->VarProp_D);
/*	DoubleVec2Screen(varSelect->VarProp_D, D);*/
	free(vpd);

	// Proportion of overall variance which is within group variance (J items).
	if(CnsIndx > 0){
		for(i=0; i<CnsIndx; i++){
			varSelect->VarProp_J[i] = varSelect->VarProp_D[i];
			varSelect->keep_D[i] = (varSelect->VarProp_J[i] < PropTol_cns);
			varSelect->keep_J[i] = (varSelect->VarProp_J[i] < PropTol_cns);
		} // i
	}

	if(OrdIndx!=CnsIndx){
		for(i=CnsIndx; i<OrdIndx; i++){
			varSelect->VarProp_J[i] = varSelect->VarProp_D[i];
			varSelect->keep_D[i] = (varSelect->VarProp_J[i] < PropTol_cat);
			varSelect->keep_J[i] = (varSelect->VarProp_J[i] < PropTol_cat);
		} // i
	}

	if(OrdIndx!=J){
		for (i=OrdIndx; i<J; i++){
			double *PropVec = OnedDoubleMemAlloc(Knom[i-OrdIndx]-1);
		
			for (j=0; j<(Knom[i-OrdIndx]-1); j++)
				PropVec[j] = varSelect->VarProp_D[nom_D[i-OrdIndx]+j];

			varSelect->VarProp_J[i] = min(PropVec, Knom[i-OrdIndx]-1);
		
			varSelect->keep_J[i] = (varSelect->VarProp_J[i] < PropTol_cat);

			for (j=0; j<(Knom[i-OrdIndx]-1); j++)
				varSelect->keep_D[nom_D[i-OrdIndx]+j] = (varSelect->VarProp_J[i] < PropTol_cat);
		
			free(PropVec);
		} // i
	}

/*	DoubleVec2Screen(varSelect->VarProp_J, J);*/
/*	printf("\n");*/
/*	DoubleVec2Screen(varSelect->VarProp_D, D);*/
/*	printf("\n");*/
/*	IntVec2Screen(varSelect->keep_J, J);*/
/*	printf("\n");*/
/*	IntVec2Screen(varSelect->keep_D, D);*/

	free(var_o);
	free(Sdiff2_w);
	free(Sdiff2_o);
	freeDoubleMat(Zbar, G);
	freeDoubleMat(diff2_w, N);
}


void VarClean(int *G, int *Q, int *N, int *J, int *D, int *CnsIndx, int *OrdIndx, int *nOrd, int *nNom, int *maxKord, double **Kord, double **Knom, int **nom_D, int **nom_J, double ***GamMat, double ***Id, int **VarNums, struct params *pars,  struct variableSelect *varSelect){
	int j, g, d;

	// Model indices
	int J_new = sumBin(varSelect->keep_J, *J);
	int CnsIndx_new = sumBin(varSelect->keep_J, *CnsIndx);
	int OrdIndx_new = sumBin(varSelect->keep_J, *OrdIndx);
	int nNom_new = J_new - OrdIndx_new;
	int nOrd_new = OrdIndx_new - CnsIndx_new;
printf("OrdIndx_new \t %d \n CnsIndx_new \t %d \n", OrdIndx_new, CnsIndx_new);
	//Variable numbers
	int *Num_tmp = OnedIntMemAlloc(J_new);
	int c=0;
	for(j=0; j<*J; j++){
		if(varSelect->keep_J[j]){
			Num_tmp[c] = (*VarNums)[j];
			c++;
		}
	} // j
	free(*VarNums);
	*VarNums = OnedIntMemAlloc(J_new);
	IntVecCopy(Num_tmp, J_new, *VarNums);
	free(Num_tmp);

	// Kord
	if(nOrd_new > 0){
		int *keep_O = OnedIntMemAlloc(*nOrd);
		for(j=*CnsIndx; j<*OrdIndx; j++)
			keep_O[j-*CnsIndx] = varSelect->keep_J[j];

		double *Ktemp_O = SubVec(*Kord, *nOrd, keep_O);
		free(*Kord);
		*Kord = OnedDoubleMemAlloc(nOrd_new);
		DoubleVecCopy(Ktemp_O, nOrd_new, *Kord);
		free(Ktemp_O);
		free(keep_O);
	}else{
		free(*Kord);
		*Kord = OnedDoubleMemAlloc(1);	
	}

	// Knom
	if(nNom_new > 0){
		int *keep_N = OnedIntMemAlloc(*nNom);
		for(j=*OrdIndx; j<*J; j++)
			keep_N[j-*OrdIndx] = varSelect->keep_J[j];
	
		double *Ktemp_N = SubVec(*Knom, *nNom, keep_N);
		free(*Knom);
		*Knom = OnedDoubleMemAlloc(nNom_new);
		DoubleVecCopy(Ktemp_N, nNom_new, *Knom);
		free(keep_N);
		free(Ktemp_N);
	}else{
		free(*Knom);
		*Knom = OnedDoubleMemAlloc(1);
	}

	int D_new = J_new + sum(*Knom, nNom_new) - 2*nNom_new;

	// nom_D
	if(nNom_new > 0){
		double Knom_sum = 0;
		free(*nom_D);
		*nom_D = OnedIntMemAlloc(nNom_new);
		for(d=0; d<nNom_new; d++){
			if(d>0){
				Knom_sum += (*Knom)[d-1];
				(*nom_D)[d] = (int)(OrdIndx_new + Knom_sum - d);
			}else{
				(*nom_D)[d] = OrdIndx_new;
			}
		} // d

		// nom_J
		free(*nom_J);
		*nom_J = OnedIntMemAlloc(nNom_new);
		for(j=OrdIndx_new; j<J_new; j++)
			(*nom_J)[j-OrdIndx_new] = j;
	}else{
		free(*nom_J);
		free(*nom_D);
		*nom_J = OnedIntMemAlloc(1);
		*nom_D = OnedIntMemAlloc(1);
	}

	// Model Parameters
	struct params temp_par, *temp_pars;
	temp_pars = &temp_par;
	allocate_params(temp_pars, *G, D_new, *Q, *N, J_new);

	for(g=0; g<*G; g++){
		SubMat(pars->lambda[g], *D, *Q+1, varSelect->keep_D, 1, temp_pars->lambda[g], D_new, *Q+1);
		SubMat(pars->A[g], *D, *Q, varSelect->keep_D, 1, temp_pars->A[g], D_new, *Q);
		SubMat(pars->Mu[g], *N, *D, varSelect->keep_D, 2, temp_pars->Mu[g], *N, D_new);
	} // g
	SubMat(pars->Y, *N, *J, varSelect->keep_J, 2, temp_pars->Y, *N, J_new);
	SubMat(pars->Z, *N, *D, varSelect->keep_D, 2, temp_pars->Z, *N, D_new);

	double **temp_ipsi = TwodDoubleMemAlloc(D_new, *D);
	SubMat(pars->inv_psi, *D, *D, varSelect->keep_D, 1, temp_ipsi, D_new, *D);
	SubMat(temp_ipsi, D_new, *D, varSelect->keep_D, 2, temp_pars->inv_psi, D_new, D_new);
	freeDoubleMat(temp_ipsi, D_new);
	DoubleMatCopy(pars->theta, *N, *Q, temp_pars->theta);
	DoubleMatCopy(pars->theta_til, *N, *Q+1, temp_pars->theta_til);
	IntMatCopy(pars->Alloc, *N, *G, temp_pars->Alloc);
	DoubleVecCopy(pars->pi_vec, *G, temp_pars->pi_vec);
	DoubleVecCopy(pars->ind, *N, temp_pars->ind);
	DoubleVecCopy(pars->ng, *G, temp_pars->ng);	

	free_params(pars, *G, *D, *Q, *N, *J);
	allocate_params(pars, *G, D_new, *Q, *N, J_new);
	copy_params(temp_pars, pars, *G, D_new, *Q, *N, J_new);
	free_params(temp_pars, *G, D_new, *Q, *N, J_new);

	freeDoubleMat(*Id, *D);
	*Id = Identity(D_new);
	
	// Threshold parameters
	if (OrdIndx_new!=CnsIndx_new){
		double **GamMat_tmp1 = TwodDoubleMemAlloc(OrdIndx_new, *maxKord+1);
		SubMat(*GamMat, *OrdIndx, *maxKord+1, varSelect->keep_J, 1, GamMat_tmp1, OrdIndx_new, *maxKord+1); // check keep
		freeDoubleMat(*GamMat, *OrdIndx);
		int maxKord_new = (int)max(*Kord, nOrd_new);
		*GamMat = TwodDoubleMemAlloc(OrdIndx_new, maxKord_new+1);
		// Copy RELEVANT rows and columns back to GamMat
		DoubleMatCopy(GamMat_tmp1, OrdIndx_new, maxKord_new+1, *GamMat);
		freeDoubleMat(GamMat_tmp1, OrdIndx_new);
		*maxKord = maxKord_new;
	}else{
		freeDoubleMat(*GamMat, *OrdIndx);
		if(OrdIndx_new>0){
			*GamMat = TwodDoubleMemAlloc(OrdIndx_new, 1);
		}else{
			*GamMat = TwodDoubleMemAlloc(1, 1);
		}
		*maxKord = 0;
	}

	*D = D_new;
	*J = J_new;
	*OrdIndx = OrdIndx_new;
	*CnsIndx = CnsIndx_new;
	*nNom = nNom_new;
	*nOrd = nOrd_new;
}

