package com.owenlynch.eniac.maths;

public class Maths {
	
    // Used by log() and exp() methods
	private static final float LOWER_BOUND = 0.9999999f;
	private static final float UPPER_BOUND = 1.0f;
	static final double ln2       = 0.69314718055995;
	
	  // Private because it only works when -1 < x < 1 but it is used by atan2
	   private static double ArcTan(double x)  // Using a Chebyshev-Pade approximation
	   {
	     double x2=x*x;
	     return (0.7162721433f+0.2996857769f*x2)*x/(0.7163164576f+(0.5377299313f+0.3951620469e-1f*x2)*x2);
	   }
	   
	   /**
	    * Arc tangent function. Thanks to Paulo Costa for donating the code.
	    */
	    public static double atan(double x)
	    {
	      return atan2(x,1);
	    }
	    
	    /**
		* Natural log function.  Returns log(a) to base E
		* Replaced with an algorithm that does not use exponents and so
		* works with large arguments.
		* @see <a href="http://www.geocities.com/zabrodskyvlada/aat/a_contents.html">here</a>
		*/
		public static double log(double x)
		{
		        if (x < 1.0)
		                return -log(1.0/x);
		                
		        double m=0.0;
		        double p=1.0;
		        while (p <= x) {
		                m++;
		                p=p*2;
		        }
		        
		        m = m - 1;
		        double z = x/(p/2);
		        
		        double zeta = (1.0 - z)/(1.0 + z);
		        double n=zeta;
		        double ln=zeta;
		        double zetasup = zeta * zeta;
		        
		        for (int j=1; true; j++)
		        {
		                n = n * zetasup;
		                double newln = ln + n / (2 * j + 1);
		                double term = ln/newln;
		                if (term >= LOWER_BOUND && term <= UPPER_BOUND)
		                        return m * ln2 - 2 * ln;
		                ln = newln;
		        }
		}

	     /**
	    * Arc tangent function valid to the four quadrants
	    * y and x can have any value without sigificant precision loss
	    * atan2(0,0) returns 0. Thanks to Paulo Costa for donating the code.
	    */
	    public static double atan2(double y, double x)
	    {
	      float ax=(float)Math.abs(x);
	      float ay=(float)Math.abs(y);

	      if ((ax<1e-7) && (ay<1e-7)) return 0f;

	      if (ax>ay)
	      {
	        if (x<0)
	        {
	          if (y>=0) return ArcTan(y/x)+Math.PI;
	          else return ArcTan(y/x)-Math.PI;
	        }
	        else return ArcTan(y/x);
	      }
	      else
	      {
	        if (y<0) return ArcTan(-x/y)-Math.PI/2;
	        else return ArcTan(-x/y)+Math.PI/2;
	      }
	    }
	    
		public static double exp(double a)
		{
		    boolean neg = a < 0 ? true : false;
		    if (a < 0)
	                a = -a;
	            int fac = 1;
	    	    double term = a;
	    	    double sum = 0;
	    	    double oldsum = 0;
	    	    double end;

	    	    do {
	    	        oldsum = sum;
	    	        sum += term;
	    
	    	        fac++;
	        
	    	        term *= a/fac;
		        end = sum/oldsum;
	      	    } while (end < LOWER_BOUND || end > UPPER_BOUND);

	            sum += 1.0f;
	            
		    return neg ? 1.0f/sum : sum;
		}
		
		/**
		* Power function.  This is a slow but accurate method.
		* Thanks to David Edwards of England for conceiving the code.
		*/
		public static double pow(double a, double b) {
			return exp(b * log(a));
		}

}
