package com.owenlynch.eniac.maths;

public class MathFunc {

	// Used by log() and exp() methods
	private static final float LOWER_BOUND = 0.9999999f;

	private static final float UPPER_BOUND = 1.0f;

	static final double ln2 = 0.69314718055995;

	// Private because it only works when -1 < x < 1 but it is used by atan2
	private static double ArcTan(double x) // Using a Chebyshev-Pade
											// approximation
	{
		double x2 = x * x;
		return (0.7162721433f + 0.2996857769f * x2)
				* x
				/ (0.7163164576f + (0.5377299313f + 0.3951620469e-1f * x2) * x2);
	}

	/**
	 * Arc tangent function. Thanks to Paulo Costa for donating the code.
	 */
	public static double atan(double x) {
		return atan2(x, 1);
	}

	/**
	 * Arc tangent function valid to the four quadrants y and x can have any
	 * value without sigificant precision loss atan2(0,0) returns 0. Thanks to
	 * Paulo Costa for donating the code.
	 */
	public static double atan2(double y, double x) {
		float ax = (float) Math.abs(x);
		float ay = (float) Math.abs(y);

		if ((ax < 1e-7) && (ay < 1e-7))
			return 0f;

		if (ax > ay) {
			if (x < 0) {
				if (y >= 0)
					return ArcTan(y / x) + Math.PI;
				else
					return ArcTan(y / x) - Math.PI;
			} else
				return ArcTan(y / x);
		} else {
			if (y < 0)
				return ArcTan(-x / y) - Math.PI / 2;
			else
				return ArcTan(-x / y) + Math.PI / 2;
		}
	}

	/**
	 * Natural log function. Returns log(a) to base E Replaced with an algorithm
	 * that does not use exponents and so works with large arguments.
	 * 
	 * @see <a
	 *      href="http://www.geocities.com/zabrodskyvlada/aat/a_contents.html">here</a>
	 */
	public static double log(double x) {
		if (x < 1.0)
			return -log(1.0 / x);

		double m = 0.0;
		double p = 1.0;
		while (p <= x) {
			m++;
			p = p * 2;
		}

		m = m - 1;
		double z = x / (p / 2);

		double zeta = (1.0 - z) / (1.0 + z);
		double n = zeta;
		double ln = zeta;
		double zetasup = zeta * zeta;

		for (int j = 1; true; j++) {
			n = n * zetasup;
			double newln = ln + n / (2 * j + 1);
			double term = ln / newln;
			if (term >= LOWER_BOUND && term <= UPPER_BOUND)
				return m * ln2 - 2 * ln;
			ln = newln;
		}
	}

	public static double exp(double a) {
		boolean neg = a < 0 ? true : false;
		if (a < 0)
			a = -a;
		int fac = 1;
		double term = a;
		double sum = 0;
		double oldsum = 0;
		double end;

		do {
			oldsum = sum;
			sum += term;

			fac++;

			term *= a / fac;
			end = sum / oldsum;
		} while (end < LOWER_BOUND || end > UPPER_BOUND);

		sum += 1.0f;

		return neg ? 1.0f / sum : sum;
	}

	/**
	 * Power function. This is a slow but accurate method. Thanks to David
	 * Edwards of England for conceiving the code.
	 */
	public static double pow(double a, double b) {
		return exp(b * log(a));
	}
	
	public static double[][] ddx(double[][] z, double Ds)
	{
		double [][] ddxz = new double[z.length][z[0].length];
		
		
		for (int i = 1 ; i < z.length-1 ; i++)
		{
			for (int j = 0 ; j < z[i].length ; j++)
			{
				ddxz[i][j] = (z[i+1][j] - z[i-1][j])/(2*Ds);
			}			
		}
		
		return ddxz;
		
	}

	public static double[][] ddy(double[][] z, double Ds)
	{
		double [][] ddyz = new double[z.length][z[0].length];

		
		for (int i = 0 ; i < z.length ; i++)
		{
			for (int j = 1 ; j < z[i].length-1 ; j++)
			{
				ddyz[i][j] = (z[i][j+1] - z[i][j-1])/(2*Ds);
			}			
		}
		
		return ddyz;
		
	}
	
	public static double[][] ddx2(double[][] z, double Ds)
	{
		double [][] ddx2z = new double[z.length][z[0].length];
		
		for (int i = 1 ; i < z.length-1 ; i++)
		{
			for (int j = 0 ; j < z[i].length ; j++)
			{
				ddx2z[i][j] = (z[i+1][j] + z[i-1][j] - 2*z[i][j])/(Ds*Ds);
			}			
		}
		
		return ddx2z;
		
	}

	public static double[][] ddy2(double[][] z, double Ds)
	{
		double [][] ddy2z = new double[z.length][z[0].length];

		
		for (int i = 0 ; i < z.length ; i++)
		{
			for (int j = 1 ; j < z[i].length-1 ; j++)
			{
				ddy2z[i][j] = (z[i][j+1] + z[i][j-1] - 2*z[i][j])/(Ds*Ds);
			}			
		}
		
		return ddy2z;		
	}	
	
	public static double[][] boundaryFill(double[][] x)
	{
		
		int a = x.length;
		int b = x[0].length;
		double[][] z = new double[a][b];
		System.arraycopy(x, 0, z, 0, x.length);
		
		for (int i = 1 ; i < a-1 ; i++)
		{	
			z[i][0] = 2*z[i][1] - z[i][2];
			z[i][b-1] = 2*z[i][b-2] - z[i][b-3];					
		}
		for (int j = 0 ; j < b ; j++)
		{	
			z[0][j] = 2*z[1][j] - z[2][j];
			z[a-1][j] = 2*z[a-2][j] - z[a-3][j];					
		}
		return z;	
	}
	
	
	public static double[][] dotStar(double[][] x, double [][] y)
	{
		int a = x.length;
		int b = x[0].length;
		double[][] z = new double[a][b];
		for (int i = 0 ; i < z.length ; i++)
		{
			for (int j = 0 ; j < z[i].length ; j++)
			{
				z[i][j] = x[i][j] * y[i][j];			
			}
		}
		return z;
	}
	
	/**
	 * 
	 * @param x
	 * @param y
	 * @return
	 */
	public static double[][] dotDivide(double[][] x, double [][] y)
	{
		int a = x.length;
		int b = x[0].length;
		double[][] z = new double[a][b];
		for (int i = 0 ; i < z.length ; i++)
		{
			for (int j = 0 ; j < z[i].length ; j++)
			{
				z[i][j] = x[i][j] / y[i][j];			
			}
		}
		return z;
	}
	
	/**
	 * matrixMult 
	 * 
	 * @param x
	 * @param y
	 * @return
	 */
	public static double[][] matrixMult(double[][] x, double [][] y)
	{
		int a = x.length;
		int b = x[0].length;
		int c = y[0].length;
		int b1 = y.length;
		if (b != b1) System.out.println("matrixMult - b != b1");
		 
		double[][] z = new double[a][c];
		
		for (int i = 0 ; i < a ; i++)
		{
			for (int j = 0 ; j < c ; j++)
			{
				z[i][j] = 0;
				for (int k = 0 ; k < b ; k++)
				{
					z[i][j] = (z[i][j] + (x[i][k] * y[k][j]));
				}
			
			}
		}
		return z;
	}
	
	public static double[][] matrixAdd(double[][] x, double [][] y)
	{
		int a = x.length;
		int b = x[0].length;
		double[][] z = new double[a][b];
		
		for (int i = 0 ; i < z.length ; i++)
		{
			for (int j = 0 ; j < z[i].length ; j++)
			{				
				z[i][j] = x[i][j] + y[i][j];			
			}
		}
		return z;		
	}
	
	
	public static double[][] matrixSub(double[][] x, double [][] y)
	{
		int a = x.length;
		int b = x[0].length;
		double[][] z = new double[a][b];
		
		for (int i = 0 ; i < z.length ; i++)
		{
			for (int j = 0 ; j < z[i].length ; j++)
			{				
				z[i][j] = x[i][j] - y[i][j];			
			}
		}
		return z;		
	}
	
	public static double matrixMax(double[][] x)
	{
		int a = x.length;
		int b = x[0].length;
		double max = -10E50;
		
		for (int i = 0 ; i < x.length ; i++)
		{
			for (int j = 0 ; j < x[i].length ; j++)
			{	
				if (x[i][j] > max)
				{
					max = x[i][j];
				}								
			}
		}
		return max;		
	}
	
	public static double matrixMin(double[][] x)
	{
		int a = x.length;
		int b = x[0].length;
		double min = 10E50;
		
		for (int i = 0 ; i < x.length ; i++)
		{
			for (int j = 0 ; j < x[i].length ; j++)
			{	
				if (x[i][j] < min)
				{
					min = x[i][j];
				}								
			}
		}
		return min;		
	}
	
	public static double[][] matrixScale(double[][] x, double c)
	{
		int a = x.length;
		int b = x[0].length;
		double[][] z = new double[a][b];
				
		for (int i = 0 ; i < a ; i++)
		{
			for (int j = 0 ; j < b ; j++)
			{				
				z[i][j] = x[i][j] * c;								
			}
		}
		return z;		
	}
	
	public static double matrixElementsSum(double[][] x)
	{
		int a = x.length;
		int b = x[0].length;
		double z = 0;
				
		for (int i = 0 ; i < a ; i++)
		{
			for (int j = 0 ; j < b ; j++)
			{				
				z = z + x[i][j];								
			}
		}
		return z;		
	}
	
	public static double[][] matrixReduce(double[][] x)
	{

		double[][] z = new double[x.length-2][x[0].length-2];
		int a = z.length;
		int b = z[0].length;
		
		for (int i = 0 ; i < z.length ; i++)
		{
			for (int j = 0 ; j < z[i].length ; j++)
			{				
				z[i][j] = x[i+1][j+1];			
			}
		}
		return z;		
	}
	
	
	public static double[][] matrixExtend(double[][] x)
	{
		int a = x.length;
		int b = x[0].length;
		double[][] z = new double[a+2][b+2];
		
		for (int i = 0 ; i < x.length ; i++)
		{
			for (int j = 0 ; j < x[i].length ; j++)
			{				
				z[i+1][j+1] = x[i][j];			
			}
		}
		return z;		
	}
	
	public static double[][] matrixInterior(double[][] x, double[][] y)
	{
		int a = x.length;
		int b = x[0].length;
		//double[][] z = new double[y.length][y[0].length];
		
		for (int i = 0 ; i < a ; i++)
		{
			for (int j = 0 ; j < b ; j++)
			{				
				y[i+1][j+1] = x[i][j];			
			}
		}
		return y;		
	}
	
	public static double sign(double x)
	{
		if (x > 0) return 1;
		else if (x == 0) return 0;
		else return -1;				
	}
}
