package com.owenlynch.eniac;

import com.owenlynch.eniac.maths.MathFunc;

public class EniacCalculator {
	// Set the parameters for the forecast (Ncase)

	//private static final int Ncase = 1; // % Forecast case number.

	private static final int DtHours = 1; // % Timestep in hours.

	private static final int M = 19; // % Points in x direction

	private static final int N = 16; // % Points in y direction

	private static final double Xp = 10; // % Coords of North Pole

	private static final double Yp = 14; // % Coords of North Pole

	private static final double Ds = 736000; // % Grid step at NP (metres)

	private static final double Centerangle = -90; // % Central angle of map.

	//private static final int MN = M * N; // % Total number of grid points.

	private static final int DAYLEN = 1; // Integration time (in days)

	//private static final int SECLEN = DAYLEN * 24 * 60 * 60; // % Integration
	// time (in
	// seconds)

	private static final double Dt = DtHours * 60 * 60; // % Timestep in seconds

	private static final int nt = DAYLEN*24/DtHours; // % Total number of time-steps.

	private static final double[] t; // % Time variable

	//private static int nspd = 24 / DtHours; // % Time steps per day

	// Define some geophysical constants (SI units)

	private static final double a = ((4 * 10000000) / (2 * Math.PI)); // Radius
	// of
	// the
	// Earth

	private static final double PI = Math.PI;

	private static final double GRAV = 9.80665; // Gravitational acceleration

	private static final double OMEGA = 2 * PI / (24 * 60 * 60); // Angular
	// velocity
	// of Earth.

	//private static final double f0 = 2 * OMEGA * Math.sin(PI / 4); // Mean
	
	// Coriolis
	// parameter.

	// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	// Compute latitude and longitude
	// on the polar stereographic grid.

	//private static final double r2d = 180 / PI; // % Conversion factor: radians
	// to degrees.

	private static final double d2r = PI / 180; // % Conversion factor: degrees
	// to radians.

	private static  double[][] LON = new double[M][N];
	private static double[][] LAT = new double[M][N];
	private static double[][] FCOR = new double[M][N];
	private static double[][] MAP = new double[M][N];
	private static double[][] h = new double[M][N];

	private static double[][] SM;
	private static double[][] SN;
	private static double[][] EIGEN;


	private double[][] zFinal;
	
	private static double[][] z0 =
	{
		{5.8201861, 5.8173748, 5.8176805, 5.8157239, 5.7962714, 5.7762526, 5.7718602, 5.8006279, 5.8138265, 5.7515187, 5.6319268, 5.4992315, 5.5164106, 5.6373589, 5.7155375, 5.7602318},
		{5.8218539, 5.8179026, 5.8141135, 5.7983247, 5.7690113, 5.7577348, 5.7703614, 5.8075907, 5.7947277, 5.6927622, 5.5332588, 5.3946908, 5.4066853, 5.5279864, 5.6137829, 5.6258647},
		{5.8261392, 5.8233765, 5.8087560, 5.7698158, 5.7351397, 5.7387305, 5.7729271, 5.8041920, 5.7610071, 5.6292046, 5.3939595, 5.2743639, 5.3088297, 5.4050452, 5.5079860, 5.4518675},
		{5.8336432, 5.8299695, 5.8037529, 5.7405566, 5.6944139, 5.6986337, 5.7353427, 5.7771332, 5.7248829, 5.5576676, 5.2737854, 5.1426808, 5.1665889, 5.2498378, 5.3752544, 5.3546069},
		{5.8404708, 5.8283508, 5.7913656, 5.7066116, 5.6424765, 5.6569862, 5.6897183, 5.7275045, 5.6881430, 5.4946749, 5.1864001, 5.0111887, 5.0361314, 5.1402362, 5.2088380, 5.2545958},
		{5.8498833, 5.8302030, 5.7843180, 5.6731162, 5.5738456, 5.6002547, 5.6357888, 5.6928216, 5.6443797, 5.4654118, 5.2182015, 4.9922897, 4.9240156, 5.0404588, 5.0485255, 5.1012187},
		{5.8608374, 5.8345042, 5.7879657, 5.6610736, 5.5203607, 5.5314779, 5.5639650, 5.6059640, 5.5723283, 5.4492604, 5.2699571, 5.0695708, 4.9383513, 4.9823770, 4.9363998, 5.0570382},
		{5.8666793, 5.8441672, 5.8049487, 5.6842722, 5.5074294, 5.3745572, 5.4102435, 5.5292799, 5.4885982, 5.3285546, 5.2275860, 5.1152839, 4.9907103, 4.8813472, 4.8294311, 5.0489879},
		{5.8697117, 5.8642772, 5.8383822, 5.7390614, 5.5787371, 5.2862156, 5.2562930, 5.4757970, 5.4434178, 5.2195024, 5.1158043, 5.0630799, 4.9870025, 4.8067789, 4.7007196, 4.9893509},
		{5.8698452, 5.8788092, 5.8707394, 5.7970723, 5.7163437, 5.5814528, 5.5289178, 5.5335433, 5.4422735, 5.2543029, 5.0792780, 5.0941046, 5.0695726, 4.9070000, 4.7810442, 5.0050608},
		{5.8772366, 5.8870516, 5.8904260, 5.8538853, 5.7847725, 5.7244481, 5.6704892, 5.5490054, 5.3740928, 5.1631855, 5.0424072, 5.0347541, 5.0110435, 4.9180578, 4.8984618, 5.0465158},
		{5.8845297, 5.8944578, 5.8964384, 5.8843815, 5.8395123, 5.7812943, 5.6965263, 5.5492693, 5.3085531, 5.1098751, 5.0234693, 5.0072800, 4.9592695, 4.9703388, 4.9653748, 5.0792854},
		{5.8779066, 5.8917422, 5.8956328, 5.8717468, 5.8264266, 5.7849867, 5.6812023, 5.5629789, 5.3960239, 5.1713576, 4.9685935, 5.0470448, 5.0275834, 5.1232335, 5.0898799, 5.1026799},
		{5.8693660, 5.8822736, 5.8833241, 5.8505751, 5.8065192, 5.7276390, 5.6206204, 5.5442351, 5.4338339, 5.2825148, 5.0824420, 5.1124639, 5.2156110, 5.2706574, 5.2091604, 5.2015096},
		{5.8619714, 5.8709371, 5.8656670, 5.8249325, 5.7720557, 5.6608548, 5.5566554, 5.4704905, 5.3794985, 5.3749333, 5.3551709, 5.3758095, 5.4519837, 5.4303051, 5.3963738, 5.4253517},
		{5.8521228, 5.8554911, 5.8570664, 5.8211887, 5.7640861, 5.6713183, 5.5712099, 5.4802329, 5.4450777, 5.4954471, 5.5630910, 5.6051932, 5.6141010, 5.4982519, 5.4893613, 5.5467581},
		{5.8449333, 5.8385985, 5.8432725, 5.8223530, 5.7792060, 5.7215654, 5.6555057, 5.6218900, 5.6301162, 5.6806457, 5.7389587, 5.7333081, 5.6862131, 5.5727064, 5.5644969, 5.6315124},
		{5.8407334, 5.8317640, 5.8292620, 5.8198157, 5.8003079, 5.7708781, 5.7399039, 5.7302078, 5.7382545, 5.7677858, 5.7907374, 5.7855732, 5.7168795, 5.6326585, 5.5797409, 5.6304285},
		{5.8337292, 5.8274791, 5.8181567, 5.8094840, 5.8029233, 5.7917329, 5.7837114, 5.7796199, 5.7788722, 5.7941299, 5.7951542, 5.7888596, 5.7252846, 5.6270603, 5.5111827, 5.5801203}
	};

	private static double[][] z24 = 
	{
		{5.8167687, 5.8075815, 5.8035281, 5.7959535, 5.7821935, 5.7781724, 5.7977050, 5.8082871, 5.7842367, 5.6646092, 5.5534528, 5.6391269, 5.7166741, 5.7735071, 5.7868656, 5.7592879},
		{5.8122744, 5.8008143, 5.7945800, 5.7716399, 5.7451469, 5.7517280, 5.7937651, 5.8275185, 5.7902640, 5.6539784, 5.5083765, 5.5625915, 5.6369762, 5.6640869, 5.6110077, 5.5167305},
		{5.8120216, 5.7952034, 5.7836914, 5.7414960, 5.7015928, 5.7120889, 5.7678601, 5.8298487, 5.7916853, 5.6372916, 5.4448490, 5.4462191, 5.5303920, 5.5121952, 5.3851265, 5.2295847},
		{5.8197429, 5.7991981, 5.7790218, 5.7293135, 5.6895880, 5.7088165, 5.7709317, 5.8336049, 5.7837272, 5.5798988, 5.3441088, 5.3192581, 5.3888472, 5.3754371, 5.2315685, 5.0719383},
		{5.8259851, 5.8107109, 5.7854060, 5.7222865, 5.6795902, 5.7075156, 5.7799244, 5.8186367, 5.7494327, 5.5244842, 5.2338134, 5.1684968, 5.2123741, 5.2522772, 5.1803225, 5.1152907},
		{5.8332615, 5.8256923, 5.7910009, 5.7118285, 5.6585015, 5.6849353, 5.7570091, 5.7913461, 5.7027710, 5.4168681, 5.0443538, 4.9914234, 5.0618703, 5.0980130, 5.0962052, 5.1145361},
		{5.8486215, 5.8384178, 5.7999787, 5.7143566, 5.6430620, 5.6564006, 5.7055702, 5.7151375, 5.6111770, 5.3276971, 5.0542186, 5.0341532, 4.9978087, 4.9889136, 4.9833260, 5.0500984},
		{5.8590047, 5.8541081, 5.8218083, 5.7210941, 5.6244787, 5.5972609, 5.5839132, 5.5752939, 5.4825058, 5.2493390, 5.0681276, 5.0481433, 4.9332723, 4.8429292, 4.7891979, 5.0045587},
		{5.8597751, 5.8668937, 5.8542719, 5.7449643, 5.6054416, 5.4709057, 5.3724757, 5.3894524, 5.3396352, 5.2354207, 5.1440298, 5.0795125, 4.9403917, 4.7648986, 4.6417725, 4.9463088},
		{5.8647679, 5.8762129, 5.8759347, 5.7982888, 5.6255625, 5.3241470, 5.1028690, 5.2220977, 5.2592086, 5.1197662, 5.0466458, 5.0789648, 5.0209868, 4.9180000, 4.7975300, 4.9388185},
		{5.8705931, 5.8814495, 5.8851987, 5.8400671, 5.7250874, 5.4999557, 5.2753661, 5.3300119, 5.3273754, 5.1624279, 5.0626693, 5.0238003, 5.0028731, 5.0075255, 4.9063930, 4.9500617},
		{5.8676273, 5.8802968, 5.8874928, 5.8614446, 5.7967246, 5.6994358, 5.5878394, 5.5179357, 5.4218088, 5.2147612, 5.0764170, 5.0181367, 4.9721444, 5.0302411, 5.0542850, 5.0455567},
		{5.8654777, 5.8782649, 5.8788132, 5.8588930, 5.8162648, 5.7516325, 5.6783270, 5.5866877, 5.4458559, 5.1672198, 5.1097696, 5.0931529, 5.0078014, 5.0502655, 5.2402987, 5.3226426},
		{5.8635895, 5.8695774, 5.8549577, 5.8244695, 5.7889911, 5.7496277, 5.6792272, 5.5725782, 5.4286817, 5.2401545, 5.2169838, 5.1674902, 5.1099523, 5.2115810, 5.4351659, 5.4986041},
		{5.8555855, 5.8509960, 5.8311528, 5.8063997, 5.7679034, 5.7152322, 5.6320282, 5.5318809, 5.4567122, 5.4258802, 5.4265159, 5.4054810, 5.4470737, 5.5694595, 5.6464786, 5.6406671},
		{5.8464414, 5.8364968, 5.8149121, 5.7905322, 5.7330946, 5.6501773, 5.5837095, 5.5300438, 5.5175803, 5.5486636, 5.6104024, 5.6582989, 5.7065070, 5.7314418, 5.6920016, 5.6924819},
		{5.8381247, 5.8259503, 5.8043523, 5.7785200, 5.7121071, 5.6253377, 5.5829632, 5.5898420, 5.6363353, 5.6989767, 5.7479087, 5.7828744, 5.7948461, 5.7093946, 5.6662962, 5.7090810},
		{5.8335655, 5.8202102, 5.8010790, 5.7836576, 5.7476366, 5.6978355, 5.6810677, 5.7009798, 5.7391969, 5.7678891, 5.7881571, 5.8017025, 5.7650119, 5.6460501, 5.6046067, 5.6182411},
		{5.8284976, 5.8194483, 5.8043787, 5.7895283, 5.7747485, 5.7612945, 5.7527108, 5.7647783, 5.7837365, 5.7888362, 5.7966285, 5.7892961, 5.7314447, 5.6300087, 5.5426557, 5.5213046}
	}
	;

	static {
		
		// initialise the static final Time variable
		t = new double[25];
		for (int i = 0; i <= nt; i++) {
			t[i] = i * Dt;
		}
		// initialise the other data
		inititialiseData();		

		// scale z0 annd z24
		z0 = MathFunc.matrixScale(z0, 1000);
		z24 = MathFunc.matrixScale(z24, 1000);		
	}

	public EniacCalculator() {
		
	}

	public EniacCalculator(double[][] initialPoints) {
		
		//z0 = initialPoints;
	}
	public void calculate()
	{
		// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
		// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
		//  Define working arrays to have correct size
		double[][] ddxz    = new double[M][N];     //  x derivative of z
		double[][] ddyz    = new double[M][N];     //  y derivative of z
		double[][] gradzsq = new double[M][N];     //  squared gradient of z
		double[][] d2dx2z  = new double[M][N];     //  second x derivative of z
		double[][] d2dy2z  = new double[M][N];     //  second y derivative of z
		double[][] xi      = new double[M][N];     //  Laplacian of z
		double[][] xi0 	= 	new double[M][N];
		double[][] eta     = new double[M][N];     //  absolute vorticity
		double[][] eta0 	= new double[M][N];
		double[][] ddxeta  = new double[M][N];     //  x derivative of eta
		double[][] ddyeta  = new double[M][N];     //  y derivative of eta
		double[][] Jacobi  = new double[M][N];     //  Jacobian J(eta,z)
		double[][] ddtz    = new double[M][N];     //  Tendency of z
		double[][] ddtxi   = new double[M][N];     //  Tendency of xi

		double[][] zdot    = new double[M-2][N-2]; //  Interior values of ddtz;
		double[][] xidot   = new double[M-2][N-2]; //  Interior values of ddtxi;
		double[][] ZDOT    = new double[M-2][N-2]; //  Fourier transform of zdot;
		double[][] XIDOT   = new double[M-2][N-2]; //  Fourier transform of xidot;
		double[] E = new double[nt+1];
		double[] S = new double[nt+1];
				
		double[][] z = new double[M][N]; //=z0; 

		// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

		
//		for (int x = 0 ; x < z.length ; x++)
//		{
//		for (int y = 0 ; y < z[0].length ; y++)
//		{
//		z[x][y] = z0[x][y];
//		}					
//		}	

		System.arraycopy(z0, 0, z, 0, z0.length);

		d2dx2z = MathFunc.ddx2(z,Ds);
		d2dy2z = MathFunc.ddy2(z,Ds);
		xi0 = MathFunc.matrixAdd(d2dx2z, d2dy2z);
		xi0 = MathFunc.boundaryFill(xi0);
		eta0 = MathFunc.dotStar(h, xi0);
		eta0 = MathFunc.matrixAdd(eta0, FCOR);

	
		// -------------------------------------------------- //

//		%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//		%%%%%%%% Start of the time-stepping loop  %%%%%%%%%
		double dt = 0;
		double[][] znm1 = new double [M][N];
		double[][] xinm1 = new double [M][N];
		double[][] znp1 = new double [M][N];
		double[][] xinp1 = new double [M][N];
		
		// ----------------------------------------------------- //
		// ------------------- Main Loop ----------------------- //
		// ----------------------------------------------------- //
		
		for (int n = 0; n < nt ; n++)
		{
			//  First time through the loop:
			if(n == 0)
			{
				dt = Dt/2;       //  First step is forward
				//znm1 = z0;      //  Copy initial height field
//				for (int x = 0 ; x < znm1.length ; x++)
//				{
//				for (int y = 0 ; y < znm1[0].length ; y++)
//				{
//				znm1[x][y] = z0[x][y];
//				}					
//				}
				System.arraycopy(z0, 0, znm1, 0, z0.length);

//				for (int x = 0 ; x < xinm1.length ; x++)
//				{
//				for (int y = 0 ; y < xinm1[0].length ; y++)
//				{
//				xinm1[x][y] = xi0[x][y];
//				}					
//				}
				System.arraycopy(xi0, 0, xinm1, 0, xi0.length);
				//xinm1 = xi0;     //  Copy initial vorticity field			    
			}

//			%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//			%%  Compute the derivatives, Laplacian and Jacobian.

//			% x-derivative of z
			ddxz = MathFunc.ddx(z, Ds);

//			% y-derivative of z
			ddyz = MathFunc.ddy(z, Ds);

//			% Square of the gradient of z
			gradzsq = MathFunc.matrixAdd(MathFunc.dotStar(ddxz, ddxz), MathFunc.dotStar(ddyz, ddyz));

//			% Second x-derivative of z
			d2dx2z = MathFunc.ddx2(z, Ds);			

//			% Second y-derivative of z
			d2dy2z = MathFunc.ddy2(z, Ds);			

//			%%  Laplacian of height
			//xi = MathFunc.matrixAdd(d2dx2z,d2dy2z);
			double[][] inner1 = MathFunc.matrixReduce(d2dx2z);
			double[][] inner2 = MathFunc.matrixReduce(d2dy2z);
			double[][] innerSum = MathFunc.matrixAdd(inner1, inner2);
			xi = MathFunc.matrixInterior(innerSum, xi);
			

//			%%  Extend xi to boundaries (to compute Jacobian).
//			%%  (First time step only; xi from BCs after that)
			if ( n == 0 )
			{
				xi = MathFunc.boundaryFill(xi);				
			}

//			%%  Absolute vorticity
			eta = MathFunc.matrixAdd(MathFunc.dotStar(h, xi),FCOR);
//			% x-derivative of eta
			ddxeta = MathFunc.ddx(eta, Ds);		

//			% y-derivative of eta
			ddyeta = MathFunc.ddy(eta, Ds);

//			%%  Compute the Jacobian J(eta,z)
			Jacobi = MathFunc.matrixSub(MathFunc.dotStar(ddxeta, ddyz),  MathFunc.dotStar(ddyeta, ddxz));


			E[n] = 0.5 * MathFunc.matrixElementsSum(gradzsq);
			S[n] = 0.5 * MathFunc.matrixElementsSum(MathFunc.dotStar(xi, xi));

//			%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//			%%  Solve the Poisson Equation Del^2(ddtz) = ddtxi
//			%%  with homogeneous boundary conditions:  z is 
//			%%  constant on the boundaries, so ddtz vanishes.


//			%% Note: Fourier transform of xidot denoted XIDOT
//			%%       Fourier transform of zdot  denoted ZDOT.
//			%%       Forward Fourier transform: 
//			%%          XIDOT = SM*xidot*SN;
//			%%       Inverse transform: 
//			%%          zdot = (4/((M-1)*(N-1)))*SM*ZDOT*SN;
//			%%

//			%  Tendency values in interior.
			xidot = MathFunc.matrixReduce(Jacobi);

//			%  Compute the transform of the solution
			double[][] work = MathFunc.matrixMult(xidot, SN);
			XIDOT = MathFunc.matrixMult(SM, work);

//			%  Convert transform of d(xi)/dt to transform of d(z)/dt
			ZDOT = MathFunc.dotDivide(XIDOT, EIGEN);

//			%  Compute inverse transform to get the height tendency.

			zdot = MathFunc.matrixMult(SM, MathFunc.matrixMult(ZDOT, SN));				
			double factor = 4/(double)((M-1)*(N-1));				
			zdot = MathFunc.matrixScale(zdot, factor);

			ddtz = MathFunc.matrixExtend(zdot);    //  Insert inner values 
			ddtxi =  MathFunc.matrixExtend(xidot); //  Insert inner values 


//			%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//			%%  Compute ddtxi on the boundaries.
//			%%  If fluid is entering through the boundary,
//			%%  we set ddtxi to zero. If fluid is leaving 
//			%%  the region, we extrapolate ddtxi linearly
//			%%  from the interior. Charney, et al., Eqn (21).

//			%  Western boundary	
			double sigW = 0;			
			for (int x = 0 ; x < ddyz[0].length ; x++)
			{
				sigW = Math.max(0, MathFunc.sign(ddyz[0][x]));
				ddtxi[0][x] = sigW*(2*ddtxi[1][x]-ddtxi[2][x]);				
			}

//			%  Eastern boundary	
			double sigE = 0;			
			for (int x = 0 ; x < ddyz[0].length ; x++)
			{
				sigE = Math.max(0, -MathFunc.sign(ddyz[M-1][x]));
				ddtxi[M-1][x] = sigE*(2*ddtxi[M-2][x]-ddtxi[M-3][x]);				
			}			

//			%  Southern boundary
			double sigS = 0;			
			for (int x = 0 ; x < ddyz.length ; x++)
			{
				sigS = Math.max(0, -MathFunc.sign(ddxz[x][0]));
				ddtxi[x][0] = sigS*(2*ddtxi[x][1]-ddtxi[x][2]);				
			}		

//			%  Northern boundary
			double sigN = 0;			
			for (int x = 0 ; x < ddyz.length ; x++)
			{
				sigN = Math.max(0, MathFunc.sign(ddxz[x][N-1]));
				ddtxi[x][N-1] = sigN*(2*ddtxi[x][N-2]-ddtxi[x][N-3]);				
			}



//			%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//			%  Step forward one time-step (leapfrog scheme).
			xinp1 = MathFunc.matrixAdd(xinm1, MathFunc.matrixScale(ddtxi, 2*dt));
			znp1  = MathFunc.matrixAdd(znm1, MathFunc.matrixScale(ddtz, 2*dt));				 

	
			//  Restore the timestep (after the first step)
			System.arraycopy(z, 0, znm1, 0, z.length);
			System.arraycopy(xi, 0, xinm1, 0, xi.length);
			System.arraycopy(znp1, 0, z, 0, znp1.length);
			System.arraycopy(xinp1, 0, xi, 0, xinp1.length);

			dt = Dt; 			

		}// end of the Main Time loop
		
		zFinal = z;
		
	}


	private static void inititialiseData(){
		//
		// Declare and initialise the variables used in loop below
		//
		// for array indexs
		int nx1 = 0;
		int ny1 = 0;
		//
		double rr = 0;
		double theta = 0.0;
		double xx = 0.0;
		double yy = 0.0;
		double lambda = 0.0;
		double phi = 0.0;
		double map = 0.0;
		double f = 0.0;
		for (int ny = 1; ny <= N; ny++) {
			for (int nx = 1; nx <= M; nx++) {
				xx = (nx - Xp) * Ds;
				yy = (ny - Yp) * Ds;
				rr = Math.sqrt((xx * xx) + (yy * yy));

				theta = MathFunc.atan2(yy, xx);
				lambda = theta + d2r * (90 + Centerangle);
				if (lambda > PI)
					lambda = lambda - 2 * PI;

				phi = 2 * ((PI / 4) - MathFunc.atan(rr / (2 * a)));
				LON[nx1][ny1] = lambda; // Longitude (radians)
				LAT[nx1][ny1] = phi; // Latitude (radians)

				nx1++;
			}
			ny1++;
			nx1=0;
		}
		// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
		// Compute Coriolis Parameter and Map Factor
		// and parameter h = g*m^2/f used in the BVE
		// (For psi-equation, h = m^2 is used).

		// reset array indexes
		ny1 = 0;
		nx1 = 0;
		for (int ny = 1; ny <= N; ny++) {
			for (int nx = 1; nx <= M; nx++) {
				lambda = LON[nx1][ny1];
				phi = LAT[nx1][ny1];
				map = 2 / (1+Math.sin(phi));
				f = 2*OMEGA*Math.sin(phi);
				MAP[nx1][ny1] = map;
				FCOR[nx1][ny1] = f;
				h[nx1][ny1] = (GRAV * (map*map) / f);	  
				nx1++;
			}
			nx1=0;
			ny1++;
		}
		// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
		// Compute Sine Matrices for the Poisson solver

		SM = new double [M-2][M-2];
		SN = new double [N-2][N-2];
		EIGEN = new double [M-2][N-2];

		// reset array indexes
		nx1=0;
		ny1=0;

		// Coefficients for x-transformation
		for (int m1 = 1 ; m1 <= M-2 ; m1++){
			for (int m2 = 1 ; m2 <= M-2 ; m2++){
				SM[nx1][ny1] = Math.sin(PI*m1*m2/(M-1));
				ny1++;
			}
			nx1++;
			ny1=0;			
		}
		// reset array indexes
		nx1=0;
		ny1=0;

		// Coefficients for y-transformation
		for (int n1 = 1 ; n1 <=  N-2 ; n1++){
			for (int n2 = 1 ; n2 <= N-2 ; n2++){
				SN[nx1][ny1] = Math.sin(PI*n1*n2/(N-1));
				ny1++;
			}
			ny1=0;
			nx1++;
		}
		// reset array indexes
		nx1=0;
		ny1=0;
		// ---------------------------------
		// Eigenvalues of Laplacian operator
		// ---------------------------------
		double eigen = 0.0;
		//temporary values to break up the calculation of eigen
		double val1 = 0.0;
		double val2 = 0.0;

		// Eigenvalues		
		for (int mm = 1 ; mm <= M-2 ; mm++){
			for (int nn = 1 ; nn <= N-2 ; nn++){				
				val1 = Math.sin((PI*mm)/(2*(M-1)));
				val2 = Math.sin((PI*nn)/(2*(N-1)));
				eigen = ((val1*val1) + (val2*val2));
				EIGEN[nx1][ny1] = (-4/(Ds*Ds)) * eigen;
				ny1++;
			}
			ny1=0;
			nx1++;
		}	
	}

	public static double[][] getZ0() {
		return z0;
	}

	public static void setZ0(double[][] z0) {
		EniacCalculator.z0 = z0;
	}

	public static double[][] getZ24() {
		return z24;
	}

	public static void setZ24(double[][] z24) {
		EniacCalculator.z24 = z24;
	}

	public double[][] getZFinal() {
		return zFinal;
	}

	public void setZFinal(double[][] final1) {
		zFinal = final1;
	}

}
